<?php

/**
 * Plugin Name: TJ Bexon Core
 * Description: Theme Junction elementor core plugin.
 * Plugin URI:  https://themejunction.net/
 * Version:     1.0.8
 * Author:      Theme Junction
 * Author URI:  https://themejunction.net/
 * Text Domain: tjcore
 * Elementor tested up to: 3.34
 * Elementor Pro tested up to: 3.34
 */

if (!defined('ABSPATH')) exit; // Exit if accessed directly

class TJCore {

	public static $_instance;

	public $plugin_name = 'TJ Core';
	public $plugin_version = '1.0.8';
	public $file = __FILE__;

	const MINIMUM_ELEMENTOR_VERSION = '3.0.0';
	const MINIMUM_PHP_VERSION = '8.1';

	public static function TJ_instance() {
		if (!isset(self::$_instance)) {
			self::$_instance = new TJCore();
		}
		return self::$_instance;
	}

	public function __construct() {
		add_action('init', array($this, 'TJ_load_textdomain'));
		add_action('plugins_loaded', array($this, 'TJ_init'));
		$this->constants();
	}

	/** ---------------------------------------------------------------
	 * Define
	 *---------------------------------------------------------------*/
	public function constants() {
		define('TJCORE_PLUGIN_NAME', $this->plugin_name);
		define('TJCORE_VERSION', $this->plugin_version);

		define('TJCORE_FILE', $this->file);
		define('TJCORE_URL', plugin_dir_url($this->file));
		define('TJCORE_DIR', dirname($this->file));
		define('TJCORE_PATH', plugin_dir_path($this->file));
		define('TJCORE_INC_PATH', TJCORE_DIR . '/inc');
		define('TJCORE_ELEMENTOR_PATH', TJCORE_DIR . '/inc/elementor');
		define('TJCORE_WIDGET_PATH', TJCORE_DIR . '/inc/widgets');
		define('TJCORE_ASSETS', TJCORE_URL . 'assets');
	}


	/** ---------------------------------------------------------------
	 * Init all hooks and others
	 *---------------------------------------------------------------*/
	public function TJ_init() {
		// include helpers for common functions
		include TJCORE_INC_PATH . '/helpers/tj-helpers.php';
		include TJCORE_INC_PATH . '/helpers/tj-elementor-trait.php';

		// Extensions
		include TJCORE_INC_PATH . '/extensions/tj-pin-element.php';

		add_action('wp_enqueue_scripts', [$this, 'tj_enqueue_frontend_js_and_css'], 20);
		add_action('admin_enqueue_scripts', [$this, 'tj_admin_enqueue_scripts']);

		add_filter('wpcf7_autop_or_not', '__return_false');

		//check elementor load
		if (did_action('elementor/loaded')) {
			// Include your theme builder
			include_once TJCORE_INC_PATH . '/theme-builder/tj-theme-builder.php';
		}

		// include one click demo importer
		include TJCORE_INC_PATH . '/demo-importer/tj-one-click-demo-import.php';

		// include custom widgets
		include TJCORE_INC_PATH . '/widgets/tj-category-widget.php';
		include TJCORE_INC_PATH . '/widgets/tj-recent-post.php';

		// include custom posts types
		include TJCORE_INC_PATH . '/custom-post/mega-menu-post.php';
		//include TJCORE_INC_PATH . '/custom-post/custom-post-services.php';
		include TJCORE_INC_PATH . '/custom-post/custom-post-projects.php';


		// Check if Elementor installed and activated
		if (!did_action('elementor/loaded')) {
			add_action('admin_notices', array($this, 'admin_notice_missing_main_plugin'));
			return;
		}

		// Check for required Elementor version
		if (!version_compare(ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=')) {
			add_action('admin_notices', array($this, 'admin_notice_minimum_elementor_version'));
			return;
		}

		// Check for required PHP version
		if (version_compare(PHP_VERSION, self::MINIMUM_PHP_VERSION, '<')) {
			add_action('admin_notices', array($this, 'admin_notice_minimum_php_version'));
			return;
		}

		// Once we get here, We have passed all validation checks so we can safely include our plugin
		require_once('plugin.php');
	}

	/** ---------------------------------------------------------------
	 * Load Plugin Textdomain
	 *---------------------------------------------------------------*/
	public function TJ_load_textdomain() {
		load_plugin_textdomain('tjcore', false, dirname($this->file) . '/languages');
	}

	/** ---------------------------------------------------------------
	 * JS and CSS for Frontend
	 *---------------------------------------------------------------*/
	function tj_enqueue_frontend_js_and_css() {
		// CSS
		wp_enqueue_style('tjcore-style', plugin_dir_url($this->file) . 'assets/css/tjcore.css', [], TJCORE_VERSION);

		// JS
		wp_enqueue_script('tj-addons-ex', plugin_dir_url($this->file) . 'inc/assets/js/tj-core-ex.js', [], TJCORE_VERSION, true);
		wp_enqueue_script('tjcore-script', plugin_dir_url($this->file) . 'assets/js/tjcore.js', ['jquery'], TJCORE_VERSION, true);
	}

	/** ---------------------------------------------------------------
	 * JS and CSS for Admin
	 *---------------------------------------------------------------*/
	public function tj_admin_enqueue_scripts() {
		if (function_exists('wp_enqueue_media')) {
			wp_enqueue_media();
		} else {
			wp_enqueue_style('thickbox');
			wp_enqueue_script('media-upload');
			wp_enqueue_script('thickbox');
		}

		wp_enqueue_style('tjcore-assistance', plugin_dir_url($this->file) . 'assets/css/admin_style.css', false);
		wp_enqueue_script('tjcore-admin', plugin_dir_url($this->file) . 'assets/js/tj_admin.js', false);
	}

	/** ---------------------------------------------------------------
	 * Admin notice
	 * Warning when the site doesn't have Elementor installed or activated.
	 *---------------------------------------------------------------*/
	public function admin_notice_missing_main_plugin() {
		if (isset($_GET['activate'])) {
			unset($_GET['activate']);
		}

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor */
			esc_html__('"%1$s" requires "%2$s" to be installed and activated.', 'tjcore'),
			'<strong>' . esc_html__('TJ Bexon Core', 'tjcore') . '</strong>',
			'<strong>' . esc_html__('Elementor', 'tjcore') . '</strong>'
		);

		printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
	}

	/** ---------------------------------------------------------------
	 * Admin notice
	 * Warning when the site doesn't have a minimum required Elementor version.
	 *---------------------------------------------------------------*/
	public function admin_notice_minimum_elementor_version() {
		if (isset($_GET['activate'])) {
			unset($_GET['activate']);
		}

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */
			esc_html__('"%1$s" requires "%2$s" version %3$s or greater.', 'tjcore'),
			'<strong>' . esc_html__('TJ Bexon Core', 'tjcore') . '</strong>',
			'<strong>' . esc_html__('Elementor', 'tjcore') . '</strong>',
			self::MINIMUM_ELEMENTOR_VERSION
		);

		printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
	}

	/** ---------------------------------------------------------------
	 * Admin notice
	 * Warning when the site doesn't have a minimum required PHP version.
	 *---------------------------------------------------------------*/
	public function admin_notice_minimum_php_version() {
		if (isset($_GET['activate'])) {
			unset($_GET['activate']);
		}

		$message = sprintf(
			/* translators: 1: Plugin name 2: PHP 3: Required PHP version */
			esc_html__('"%1$s" requires "%2$s" version %3$s or greater.', 'tjcore'),
			'<strong>' . esc_html__('ThemeJunction Core', 'tjcore') . '</strong>',
			'<strong>' . esc_html__('PHP', 'tjcore') . '</strong>',
			self::MINIMUM_PHP_VERSION
		);

		printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
	}
}
TJCore::TJ_instance();
