<?php

namespace TJCore;

use Elementor\Controls_Manager;
use Elementor\Widget_Base;
use \Elementor\Group_Control_Typography;

if (!defined('ABSPATH')) {
  exit;
}
// Exit if accessed directly

/**
 * Elementor widget for Heading.
 */
class TJ_Heading extends Widget_Base {

  // widget name
  public function get_name() {
    return 'tj-heading';
  }

  // widget title.
  public function get_title() {
    return __('TJ Heading', 'tjcore');
  }

  // widget icon.
  public function get_icon() {
    return 'eicon-heading tj-icon';
  }

  // Widget categories.
  public function get_categories() {
    return ['tjcore'];
  }

  // scripts dependencies.
  public function get_script_depends() {
    return ['tjcore-script'];
  }

  // styles dependencies.
  public function get_style_depends() {
    return ['tjcore-style'];
  }

  // Widget keywords.
  public function get_keywords() {
    return [
      'heading',
      'headings',
      'tj heading',
      'tj',
      'tj addons',
      'tjcore',
    ];
  }

  // Widget help url.
  public function get_custom_help_url() {
    return 'https://go.elementor.com/';
  }

  /**
   * widget controls.
   */
  protected function register_controls() {

    // tj_title_tab
    $this->start_controls_section(
      'tj_title_tab',
      [
        'label' => esc_html__('Heading', 'tjcore'),
        'tab'   => Controls_Manager::TAB_CONTENT,
      ]
    );
    $this->add_control(
      'tj_title',
      [
        'label'       => esc_html__('Title', 'tjcore'),
        'type'        => Controls_Manager::TEXTAREA,
        'rows'        => 5,
        'default'     => esc_html__('TJ heading title.', 'tjcore'),
        'placeholder' => esc_html__('Type heading title here.', 'tjcore'),
        'show_label'  => false,
        'dynamic'     => [
          'active' => true,
        ],
      ]
    );
    $this->add_control(
      'tj_title_link',
      [
        'label'   => esc_html__('Link', 'tjcore'),
        'type'    => Controls_Manager::URL,
        'dynamic' => [
          'active' => true,
        ],
      ]
    );

    $this->add_control(
      'tj_title_tag',
      [
        'label'   => esc_html__('Title Tags', 'tjcore'),
        'type'    => Controls_Manager::CHOOSE,
        'options' => [
          'h1' => [
            'title' => esc_html__('H1', 'tjcore'),
            'icon'  => 'eicon-editor-h1',
          ],
          'h2' => [
            'title' => esc_html__('H2', 'tjcore'),
            'icon'  => 'eicon-editor-h2',
          ],
          'h3' => [
            'title' => esc_html__('H3', 'tjcore'),
            'icon'  => 'eicon-editor-h3',
          ],
          'h4' => [
            'title' => esc_html__('H4', 'tjcore'),
            'icon'  => 'eicon-editor-h4',
          ],
          'h5' => [
            'title' => esc_html__('H5', 'tjcore'),
            'icon'  => 'eicon-editor-h5',
          ],
          'h6' => [
            'title' => esc_html__('H6', 'tjcore'),
            'icon'  => 'eicon-editor-h6',
          ],
        ],
        'default' => 'h2',
        'toggle'  => false,
      ]
    );
    $this->add_responsive_control(
      'tj_align',
      [
        'label'     => esc_html__('Alignment', 'tjcore'),
        'type'      => Controls_Manager::CHOOSE,
        'options'   => [
          'start'   => [
            'title' => esc_html__('Left', 'tjcore'),
            'icon'  => 'eicon-text-align-left',
          ],
          'center' => [
            'title' => esc_html__('Center', 'tjcore'),
            'icon'  => 'eicon-text-align-center',
          ],
          'end'  => [
            'title' => esc_html__('Right', 'tjcore'),
            'icon'  => 'eicon-text-align-right',
          ],
        ],
        'default'   => 'start',
        'toggle'    => false,
        'selectors' => [
          '{{WRAPPER}} .tj_heading' => 'text-align: {{VALUE}};',
        ],
      ]
    );

    // animation
    $this->add_control(
      'tj_gsap_animation',
      [
        'label'        => esc_html__('Enable GSAP Animation?', 'tjcore'),
        'type'         => Controls_Manager::SWITCHER,
        'label_on'     => esc_html__('Yes', 'tjcore'),
        'label_off'    => esc_html__('No', 'tjcore'),
        'return_value' => 'yes',
        'default'      => '0',
      ]
    );
    $this->add_control(
      'animation_type',
      [
        'label' => esc_html__('Animation type', 'tjcore'),
        'type' => Controls_Manager::SELECT,
        'default' => 'title-anim',
        'options' => [
          'title-anim' => esc_html__('Split Text 1', 'tjcore'),
          'text-anim' => esc_html__('Split Text 2', 'tjcore'),
          'title-highlight'  => esc_html__('Title Highlight', 'tjcore'),
        ],
        'condition' => [
          'tj_gsap_animation' => 'yes'
        ]
      ]
    );
    $this->add_control(
      'tj_animation',
      [
        'label'        => esc_html__('Enable Animation?', 'tjcore'),
        'type'         => Controls_Manager::SWITCHER,
        'label_on'     => esc_html__('Yes', 'tjcore'),
        'label_off'    => esc_html__('No', 'tjcore'),
        'return_value' => 'yes',
        'default'      => '0',
        'separator'    => 'before',
      ]
    );
    $this->add_control(
      'tj_animation_name',
      [
        'label'     => esc_html__('Animation Name', 'tjcore'),
        'type'      => Controls_Manager::SELECT,
        'default'   => '',
        'options'   => [
          ''            => esc_html__('Default', 'tjcore'),
          'zoomIn'      => esc_html__('Zoom In', 'tjcore'),
          'fadeInUp'    => esc_html__('Fade In Up', 'tjcore'),
          'fadeInDown'  => esc_html__('Fade In Down', 'tjcore'),
          'fadeInLeft'  => esc_html__('Fade In Left', 'tjcore'),
          'fadeInRight' => esc_html__('Fade In Right', 'tjcore'),
        ],
        'condition' => [
          'tj_animation' => 'yes',
        ],
      ]
    );
    $this->add_control(
      'tj_animation_delay',
      [
        'label'      => esc_html__('Animation Delay', 'tjcore'),
        'type'       => Controls_Manager::SLIDER,
        'size_units' => ['px'],
        'range'      => [
          'px' => [
            'min'  => 0,
            'max'  => 3,
            'step' => .1,
          ],
        ],
        'default'    => [
          'unit' => 'px',
          'size' => 0.3,
        ],
        'condition'  => [
          'tj_animation' => 'yes',
        ],
      ]
    );
    $this->end_controls_section();

    /** ---------------------------------------------------------------
     * TAB_STYLE
     * -------------------------------------------------------------* */
    $this->start_controls_section(
      'section_heading_style',
      [
        'label' => esc_html__('Heading', 'tjcore'),
        'tab'   => Controls_Manager::TAB_STYLE,
      ]
    );
    $this->add_group_control(
      Group_Control_Typography::get_type(),
      [
        'name'     => 'text_typography',
        'label'    => esc_html__('Typography', 'tjcore'),
        'selector' => '{{WRAPPER}} .tj_heading .title',
      ]
    );
    $this->add_control(
      'text_color',
      [
        'label'     => esc_html__('Color', 'tjcore'),
        'type'      => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .tj_heading .title' => 'color: {{VALUE}}',
        ],
      ]
    );
    $this->add_control(
      'text_highlight_color',
      [
        'label'     => esc_html__('Highlighter Color', 'tjcore'),
        'type'      => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .tj_heading .title span' => 'color: {{VALUE}}',
        ],
      ]
    );
    $this->add_responsive_control(
      'text_editor_padding',
      [
        'label'      => esc_html__('Padding', 'tjcore'),
        'type'       => Controls_Manager::DIMENSIONS,
        'size_units' => ['px', '%', 'em'],
        'selectors'  => [
          '{{WRAPPER}} .tj_heading .title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
        ],
      ]
    );
    $this->add_responsive_control(
      'text_editor_margin',
      [
        'label'      => esc_html__('Margin', 'tjcore'),
        'type'       => Controls_Manager::DIMENSIONS,
        'size_units' => ['px', '%', 'em'],
        'selectors'  => [
          '{{WRAPPER}} .tj_heading .title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
        ],
      ]
    );
    $this->end_controls_section();
  }

  /**
   * Render the widget output on the frontend.
   */
  protected function render() {
    $settings = $this->get_settings_for_display();

    // Extract settings with fallbacks
    $title            = $settings['tj_title'] ?? '';
    $title_tag        = $settings['tj_title_tag'] ?? 'h2';
    $title_link       = $settings['tj_title_link']['url'] ?? '';
    $is_external      = !empty($settings['tj_title_link']['is_external']) ? ' target="_blank"' : '';
    $nofollow         = !empty($settings['tj_title_link']['nofollow']) ? ' rel="nofollow"' : '';

    $enable_gsap      = $settings['tj_gsap_animation'] ?? '';
    $animation_type   = $settings['animation_type'] ?? '';
    $enable_animation = $settings['tj_animation'] ?? '';
    $animation_name   = $settings['tj_animation_name'] ?? '';
    $animation_delay  = $settings['tj_animation_delay']['size'] ?? '';

    // Determine wrapper classes
    $wrapper_classes = 'tj_heading';

    if (!empty($enable_gsap) && !empty($animation_type)) {
      $wrapper_classes .= ' tj-title ' . esc_attr($animation_type);
    } elseif (!empty($enable_animation) && !empty($animation_name)) {
      $wrapper_classes .= ' tj-title wow ' . esc_attr($animation_name);

      if (!empty($animation_delay)) {
        $this->add_render_attribute('tj-heading-wrapper', 'data-wow-delay', esc_attr($animation_delay . 's'));
      }
    }

    $this->add_render_attribute('tj-heading-wrapper', 'class', esc_attr($wrapper_classes));
    $this->add_render_attribute('tj-heading-title', 'class', 'title');

    echo '<div ' . $this->get_render_attribute_string('tj-heading-wrapper') . '>';

    if (!empty($title_link)) {
      echo sprintf(
        '<a href="%1$s"%2$s%3$s>',
        esc_url($title_link),
        $is_external,
        $nofollow
      );
    }

    printf(
      '<%1$s %2$s>%3$s</%1$s>',
      tag_escape($title_tag),
      $this->get_render_attribute_string('tj-heading-title'),
      tj_kses($title)
    );

    if (!empty($title_link)) {
      echo '</a>';
    }

    echo '</div>';
  }

  protected function content_template() {
  }
}
$widgets_manager->register(new TJ_Heading());
