<?php

namespace TJCore;

use \Elementor\Modules\DynamicTags\Module as TagsModule;
use \Elementor\Widget_Base;
use \Elementor\Repeater;
use Elementor\Controls_Manager;

if (! defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor google maps widget.
 */
class TJ_Google_Maps extends Widget_Base
{

	use \TJCore\TJCore_Elelmentor_Trait;

	/**
	 * Get widget name.
	 */
	public function get_name()
	{
		return 'tj-google_maps';
	}

	/**
	 * Get widget title.
	 */
	public function get_title()
	{
		return esc_html__('TJ Maps', 'tjcore');
	}

	/**
	 * Get widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-google-maps tj-icon';
	}


	/**
	 * Widget categories.
	 */
	public function get_categories()
	{
		return ['tjcore'];
	}

	// scripts dependencies.
	public function get_script_depends()
	{
		return ['tjcore-script'];
	}

	// styles dependencies.
	public function get_style_depends()
	{
		return ['tjcore-style'];
	}

	// Widget keywords.
	public function get_keywords()
	{
		return [
			'map',
			'maps',
			'tj google maps',
			'tj',
			'tj addons',
			'tjcore',
		];
	}

	// Widget help url.
	public function get_custom_help_url()
	{
		return 'https://go.elementor.com/';
	}

	/**
	 * widget controls.
	 */
	protected function register_controls()
	{


		$this->start_controls_section(
			'tj_design_tab',
			[
				'label' => esc_html__('Map Style', 'tjcore'),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'tj_design_style',
			[
				'label'   => esc_html__('Select Style', 'tjcore'),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					1 => esc_html__('Style 01', 'tjcore'),
					2 => esc_html__('Style 02', 'tjcore'),
				],
				'default' => 1,
			]
		);
		$this->end_controls_section();


		$this->start_controls_section(
			'section_map',
			[
				'label' => esc_html__('Google Maps', 'tjcore'),
				'condition'        => [
					'tj_design_style!' => ['1'],
				],
			]
		);

		if (\Elementor\Plugin::$instance->editor->is_edit_mode()) {
			$api_key = get_option('elementor_google_maps_api_key');

			if (! $api_key) {
				$this->add_control(
					'api_key_notification',
					[
						'type' => Controls_Manager::ALERT,
						'alert_type' => 'info',
						'content' => sprintf(
							/* translators: 1: Integration settings link open tag, 2: Create API key link open tag, 3: Link close tag. */
							esc_html__('Set your Google Maps API Key in Elementor\'s %1$sIntegrations Settings%3$s page. Create your key %2$shere.%3$s', 'tjcore'),
							'<a href="' . \Elementor\Settings::get_settings_tab_url('integrations') . '" target="_blank">',
							'<a href="https://developers.google.com/maps/documentation/embed/get-api-key" target="_blank">',
							'</a>'
						),
					]
				);
			}
		}

		$default_address = esc_html__('London Eye, London, United Kingdom', 'tjcore');
		$this->add_control(
			'address',
			[
				'label' => esc_html__('Location', 'tjcore'),
				'type' => Controls_Manager::TEXT,
				'dynamic' => [
					'active' => true,
					'categories' => [
						TagsModule::POST_META_CATEGORY,
					],
				],
				'ai' => [
					'active' => false,
				],
				'placeholder' => $default_address,
				'default' => $default_address,
				'label_block' => true,
			]
		);

		$this->add_control(
			'zoom',
			[
				'label' => esc_html__('Zoom', 'tjcore'),
				'type' => Controls_Manager::SLIDER,
				'default' => [
					'size' => 10,
				],
				'range' => [
					'px' => [
						'min' => 1,
						'max' => 20,
					],
				],
				'separator' => 'before',
			]
		);

		$this->add_responsive_control(
			'height',
			[
				'label' => esc_html__('Height', 'tjcore'),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 40,
						'max' => 1440,
					],
				],
				'size_units' => ['px', 'em', 'rem', 'vh', 'custom'],
				'selectors' => [
					'{{WRAPPER}} iframe' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_map_content',
			[
				'label' => esc_html__('Maps Content', 'tjcore'),
				'condition'        => [
					'tj_design_style!' => ['2', '3', '4'],
				],
			]
		);

		$this->add_control(
			'tj_image',
			[
				'label'   => esc_html__('Choose Image', 'tjcore'),
				'type'    => Controls_Manager::MEDIA,
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);

		$repeater = new Repeater();
		$repeater->add_control(
			'tj_class_name',
			[
				'label'       => esc_html__('Class Name', 'tjcore'),
				'type'        => Controls_Manager::TEXT,
				'description' => esc_html__('We have only 3 class name loc-1, loc-2, loc-3', 'tjcore'),
				'default'     => tj_kses('loc-1'),
			]
		);
		$repeater->add_control(
			'tj_map_content',
			[
				'label'       => esc_html__('Content Area', 'tjcore'),
				'type'        => Controls_Manager::TEXTAREA,
				'default'     => tj_kses('<span>Head office:</span><p>993 Renner Burg, West Rond, MT 94251-030, USA.</p><a href="tel:10095447818">P: +1 (009) 544-7818</a>
                    					<a href="mailto:support@bexon.com">M: support@bexon.com</a>'),
				'rows'        => '8',
			]
		);

		$this->add_control(
			'tj_content_list',
			[
				'label'    => esc_html__('Map Content', 'tjcore'),
				'type'     => Controls_Manager::REPEATER,
				'fields'   => $repeater->get_controls(),
				'default'  => [
					[
						'tj_map_content' => tj_kses('<span>Head office:</span><p>993 Renner Burg, West Rond, MT 94251-030, USA.</p><a href="tel:10095447818">P: +1 (009) 544-7818</a>
                    								<a href="mailto:support@bexon.com">M: support@bexon.com</a>'),
					],
					[
						'tj_map_content' => tj_kses('<span>Regional office:</span><p>Hessisch Lichtenau 37235, Kassel, Germany.</p><a href="tel:10098801810">P: +1 (009) 880-1810</a>
                    								<a href="mailto:support@bexon.com">M: support@bexon.com</a>'),
					],
					[
						'tj_map_content' => tj_kses('<span>Regional office:</span><p>32 Altamira, State of Pará, Brazil.</p><a href="tel:10095447818">P: +1 (009) 544-7818</a>
                    								<a href="mailto:support@bexon.com">M: support@bexon.com</a>'),
					]
				],
				'title_field' => '{{{ tj_map_content }}}',
			]
		);

		$this->end_controls_section();

		// animation
		$this->start_controls_section(
			'section_animation_tab',
			[
				'label' => esc_html__('Animation', 'tjcore'),
			]
		);
		$this->add_control(
			'tj_animation',
			[
				'label'        => esc_html__('Enable Animation?', 'tjcore'),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__('Yes', 'tjcore'),
				'label_off'    => esc_html__('No', 'tjcore'),
				'return_value' => 'yes',
				'default'      => '0',
				'separator'    => 'before',
			]
		);
		$this->add_control(
			'tj_animation_name',
			[
				'label'     => esc_html__('Animation Name', 'tjcore'),
				'type'      => Controls_Manager::SELECT,
				'default'   => '',
				'options'   => [
					''            => esc_html__('Default', 'tjcore'),
					'zoomIn'      => esc_html__('Zoom In', 'tjcore'),
					'fadeInLeft'  => esc_html__('Fade In Left', 'tjcore'),
					'fadeInRight' => esc_html__('Fade In Right', 'tjcore'),
					'fadeInUp'    => esc_html__('Fade In Up', 'tjcore'),
					'fadeInDown'  => esc_html__('Fade In Down', 'tjcore'),
				],
				'condition' => [
					'tj_animation' => 'yes',
				],
			]
		);
		$this->add_control(
			'tj_animation_delay',
			[
				'label'      => esc_html__('Animation Delay', 'tjcore'),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => ['px'],
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 3,
						'step' => .1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 0.3,
				],
				'condition'  => [
					'tj_animation' => 'yes',
				],
			]
		);
		$this->end_controls_section();



		/** ---------------------------------------------------------------
		 * TAB_STYLE
		 * -------------------------------------------------------------* */

		$this->start_controls_section(
			'section_map_style',
			[
				'label' => esc_html__('Google Maps', 'tjcore'),
				'tab'   => Controls_Manager::TAB_STYLE,
				'condition'        => [
					'tj_design_style!' => ['1'],
				],
			]
		);

		$this->start_controls_tabs('map_filter');

		$this->start_controls_tab(
			'normal',
			[
				'label' => esc_html__('Normal', 'tjcore'),
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Css_Filter::get_type(),
			[
				'name' => 'css_filters',
				'selector' => '{{WRAPPER}} iframe',
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'hover',
			[
				'label' => esc_html__('Hover', 'tjcore'),
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Css_Filter::get_type(),
			[
				'name' => 'css_filters_hover',
				'selector' => '{{WRAPPER}}:hover iframe',
			]
		);

		$this->add_control(
			'hover_transition',
			[
				'label' => esc_html__('Transition Duration', 'tjcore') . ' (s)',
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 3,
						'step' => 0.1,
					],
				],
				'selectors' => [
					'{{WRAPPER}} iframe' => 'transition-duration: {{SIZE}}s',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

		$this->start_controls_section(
			'section_location_style',
			[
				'label' => esc_html__('Circle Style', 'tjcore'),
				'tab'   => Controls_Manager::TAB_STYLE,
				'condition'        => [
					'tj_design_style!' => ['2'],
				],
			]
		);

		$this->add_control(
			'circle_color',
			[
				'label'     => esc_html__('Color', 'tjcore'),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .location-indicator' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'circle_width',
			[
				'label' => esc_html__('Width', 'tjcore'),
				'type' => Controls_Manager::SLIDER,
				'size_units' => ['px', '%', 'em'],
				'selectors' => [
					'{{WRAPPER}} .location-indicator' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'circle_height',
			[
				'label' => esc_html__('Height', 'tjcore'),
				'type' => Controls_Manager::SLIDER,
				'size_units' => ['px', '%', 'em'],
				'selectors' => [
					'{{WRAPPER}} .location-indicator' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_location_tooltip',
			[
				'label' => esc_html__('Location Tooltip', 'tjcore'),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				'condition'        => [
					'tj_design_style!' => ['2'],
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Background::get_type(),
			[
				'name'     => 'location_tooltip_background',
				'selector' => '{{WRAPPER}} .location-tooltip',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'location_tooltip_border',
				'selector' => '{{WRAPPER}} .location-tooltip',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'location_tooltip_box_shadow',
				'selector' => '{{WRAPPER}} .location-tooltip',
			]
		);

		$this->add_responsive_control(
			'location_tooltip_padding',
			[
				'label'      => esc_html__('Padding', 'tjcore'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%', 'em'],
				'selectors'  => [
					'{{WRAPPER}} .location-tooltip' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'location_tooltip_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'tjcore'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%'],
				'selectors'  => [
					'{{WRAPPER}} .location-tooltip' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		//
		// Label text (span)
		//
		$this->add_control(
			'location_tooltip_heading_span',
			[
				'label' => esc_html__('Label (span)', 'tjcore'),
				'type'  => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'location_tooltip_span_color',
			[
				'label' => esc_html__('Text Color', 'tjcore'),
				'type'  => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .location-tooltip span' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'location_tooltip_span_typography',
				'selector' => '{{WRAPPER}} .location-tooltip span',
			]
		);

		//
		// Address (p)
		//
		$this->add_control(
			'location_tooltip_heading_paragraph',
			[
				'label' => esc_html__('Address (p)', 'tjcore'),
				'type'  => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'location_tooltip_p_color',
			[
				'label' => esc_html__('Text Color', 'tjcore'),
				'type'  => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .location-tooltip p' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'location_tooltip_p_typography',
				'selector' => '{{WRAPPER}} .location-tooltip p',
			]
		);

		//
		// Phone & Email Links (a)
		//
		$this->add_control(
			'location_tooltip_heading_links',
			[
				'label' => esc_html__('Phone & Email Links (a)', 'tjcore'),
				'type'  => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		// Start link color tabs
		$this->start_controls_tabs('tabs_location_tooltip_links');

		$this->start_controls_tab(
			'tab_location_link_normal',
			[
				'label' => esc_html__('Normal', 'tjcore'),
			]
		);

		$this->add_control(
			'location_link_color',
			[
				'label' => esc_html__('Link Color', 'tjcore'),
				'type'  => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .location-tooltip a' => 'color: {{VALUE}}',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_location_link_hover',
			[
				'label' => esc_html__('Hover', 'tjcore'),
			]
		);

		$this->add_control(
			'location_link_hover_color',
			[
				'label' => esc_html__('Hover Color', 'tjcore'),
				'type'  => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .location-tooltip a:hover' => 'color: {{VALUE}}',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs(); // End link tabs

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'location_tooltip_link_typography',
				'selector' => '{{WRAPPER}} .location-tooltip a',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render google maps widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();

		$tj_design_style = (!empty($settings['tj_design_style']) && $settings['tj_design_style'] > 0) ? $settings['tj_design_style'] : 1;

		include dirname(__FILE__) . '/styles/google-map/style-' . $tj_design_style . '.php';
	}
}

$widgets_manager->register(new TJ_Google_Maps());
