<?php

namespace TJCore;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Core\Breakpoints\Manager as Breakpoints_Manager;
use Elementor\Core\Kits\Documents\Tabs\Global_Colors;
use Elementor\Core\Kits\Documents\Tabs\Global_Typography;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Css_Filter;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Typography;
use Elementor\Repeater;
use Elementor\Utils;
use Elementor\Plugin;


if (!defined('ABSPATH')) {
  exit;
}
// Exit if accessed directly

/**
 * Elementor widget for Gallery.
 */
class TJ_Gallery extends Widget_Base {

  // widget name
  public function get_name() {
    return 'tj-gallery';
  }

  // widget title.
  public function get_title() {
    return __('TJ Gallery', 'tjcore');
  }

  // widget icon.
  public function get_icon() {
    return 'eicon-gallery-justified tj-icon';
  }

  // Widget categories.
  public function get_categories() {
    return ['tjcore'];
  }

  // scripts dependencies.
  public function get_script_depends() {
    return ['tj-gallery', 'elementor-gallery'];
  }

  // styles dependencies.
  public function get_style_depends() {
    return ['tj-gallery', 'elementor-gallery'];
  }

  // Widget keywords.
  public function get_keywords() {
    return [
      'gallery',
      'images',
      'tj gallery',
      'tj',
      'tj addons',
      'tjcore',
    ];
  }

  // Widget help url.
  public function get_custom_help_url() {
    return 'https://go.elementor.com/';
  }

  protected function is_dynamic_content(): bool {
    return false;
  }

  public function has_widget_inner_wrapper(): bool {
    return !Plugin::$instance->experiments->is_feature_active('e_optimized_markup');
  }

  /**
   * widget controls.
   */
  protected function register_controls() {

    // tj_gallery_settings
    $this->start_controls_section(
      'tj_gallery_settings',
      [
        'label' => esc_html__('Settings', 'tjcore'),
        'tab'   => Controls_Manager::TAB_CONTENT,
      ]
    );
    $this->add_control(
      'gallery',
      [
        'label' => esc_html__('Add Images', 'tjcore'),
        'type' => Controls_Manager::GALLERY,
        'dynamic' => [
          'active' => true,
        ],
      ]
    );
    $this->add_control(
      'order_by',
      [
        'type' => Controls_Manager::SELECT,
        'label' => esc_html__('Order By', 'tjcore'),
        'options' => [
          '' => esc_html__('Default', 'tjcore'),
          'random' => esc_html__('Random', 'tjcore'),
        ],
        'default' => '',
      ]
    );
    $this->add_control(
      'lazyload',
      [
        'type' => Controls_Manager::SWITCHER,
        'label' => esc_html__('Lazy Load', 'tjcore'),
        'return_value' => 'yes',
        'default' => 'yes',
        'frontend_available' => true,
      ]
    );
    $this->add_control(
      'gallery_layout',
      [
        'type' => Controls_Manager::SELECT,
        'label' => esc_html__('Layout', 'tjcore'),
        'default' => 'grid',
        'options' => [
          'grid' => esc_html__('Grid', 'tjcore'),
          'justified' => esc_html__('Justified', 'tjcore'),
          'masonry' => esc_html__('Masonry', 'tjcore'),
        ],
        'separator' => 'before',
        'frontend_available' => true,
      ]
    );
    $this->add_responsive_control(
      'columns',
      [
        'label' => esc_html__('Columns', 'tjcore'),
        'type' => Controls_Manager::NUMBER,
        'default' => 4,
        'tablet_default' => 2,
        'mobile_default' => 1,
        'min' => 1,
        'max' => 24,
        'condition' => [
          'gallery_layout!' => 'justified',
        ],
        'render_type' => 'none',
        'frontend_available' => true,
      ]
    );

    $active_breakpoints = Plugin::$instance->breakpoints->get_active_breakpoints();
    $ideal_row_height_device_args = [];
    $gap_device_args = [];

    // Add default values for all active breakpoints.
    foreach ($active_breakpoints as $breakpoint_name => $breakpoint_instance) {
      if ('widescreen' !== $breakpoint_name) {
        $ideal_row_height_device_args[$breakpoint_name] = [
          'default' => [
            'size' => 150,
          ],
        ];

        $gap_device_args[$breakpoint_name] = [
          'default' => [
            'size' => 10,
          ],
        ];
      }
    }

    $this->add_responsive_control(
      'ideal_row_height',
      [
        'label' => esc_html__('Row Height', 'tjcore'),
        'type' => Controls_Manager::SLIDER,
        'range' => [
          'px' => [
            'min' => 50,
            'max' => 500,
          ],
        ],
        'default' => [
          'size' => 200,
        ],
        'device_args' => $ideal_row_height_device_args,
        'condition' => [
          'gallery_layout' => 'justified',
        ],
        'required' => true,
        'render_type' => 'none',
        'frontend_available' => true,
      ]
    );
    $this->add_responsive_control(
      'gap',
      [
        'label' => esc_html__('Spacing', 'tjcore'),
        'type' => Controls_Manager::SLIDER,
        'default' => [
          'size' => 10,
        ],
        'device_args' => $gap_device_args,
        'required' => true,
        'render_type' => 'none',
        'frontend_available' => true,
      ]
    );
    $this->add_control(
      'link_to',
      [
        'label' => esc_html__('Link', 'tjcore'),
        'type' => Controls_Manager::SELECT,
        'default' => 'file',
        'options' => [
          '' => esc_html__('None', 'tjcore'),
          'file' => esc_html__('Media File', 'tjcore'),
          'custom' => esc_html__('Custom URL', 'tjcore'),
        ],
        'frontend_available' => true,
      ]
    );
    $this->add_control(
      'url',
      [
        'label' => esc_html__('URL', 'tjcore'),
        'type' => Controls_Manager::URL,
        'condition' => [
          'link_to' => 'custom',
        ],
        'frontend_available' => true,
        'dynamic' => [
          'active' => true,
        ],
      ]
    );
    $this->add_control(
      'open_lightbox',
      [
        'label' => esc_html__('Lightbox', 'tjcore'),
        'type' => Controls_Manager::SELECT,
        'description' => sprintf(
          /* translators: 1: Link open tag, 2: Link close tag. */
          esc_html__('Manage your site’s lightbox settings in the %1$sLightbox panel%2$s.', 'tjcore'),
          '<a href="javascript: $e.run( \'panel/global/open\' ).then( () => $e.route( \'panel/global/settings-lightbox\' ) )">',
          '</a>'
        ),
        'default' => 'default',
        'options' => [
          'default' => esc_html__('Default', 'tjcore'),
          'yes' => esc_html__('Yes', 'tjcore'),
          'no' => esc_html__('No', 'tjcore'),
        ],
        'condition' => [
          'link_to' => 'file',
        ],
        'assets' => [
          'styles' => [
            [
              'name' => 'e-swiper',
              'conditions' => [
                'terms' => [
                  [
                    'name' => 'link_to',
                    'operator' => '===',
                    'value' => 'file',
                  ],
                  [
                    'name' => 'open_lightbox',
                    'operator' => '!==',
                    'value' => 'no',
                  ],
                ],
              ],
            ],
          ],
          'scripts' => [
            [
              'name' => 'swiper',
              'conditions' => [
                'terms' => [
                  [
                    'name' => 'link_to',
                    'operator' => '===',
                    'value' => 'file',
                  ],
                  [
                    'name' => 'open_lightbox',
                    'operator' => '!==',
                    'value' => 'no',
                  ],
                ],
              ],
            ],
          ],
        ],
      ]
    );
    $this->add_control(
      'aspect_ratio',
      [
        'type' => Controls_Manager::SELECT,
        'label' => esc_html__('Aspect Ratio', 'tjcore'),
        'default' => '3:2',
        'options' => [
          '1:1' => '1:1',
          '3:2' => '3:2',
          '4:3' => '4:3',
          '9:16' => '9:16',
          '16:9' => '16:9',
          '21:9' => '21:9',
        ],
        'condition' => [
          'gallery_layout' => 'grid',
        ],
        'render_type' => 'none',
        'frontend_available' => true,
      ]
    );
    $this->add_group_control(
      Group_Control_Image_Size::get_type(),
      [
        'name' => 'thumbnail_image',
        'default' => 'medium',
      ]
    );
    $this->end_controls_section();

    // overlay
    $this->start_controls_section(
      'tj_gallery_overlay',
      [
        'label' => esc_html__('Overlay', 'tjcore')
      ]
    );
    $this->add_control(
      'overlay_background',
      [
        'label' => esc_html__('Background', 'tjcore'),
        'type' => Controls_Manager::SWITCHER,
        'default' => 'yes',
        'frontend_available' => true,
      ]
    );
    $this->add_control(
      'overlay_icon',
      [
        'label'            => esc_html__('Icon', 'tjcore'),
        'type'             => Controls_Manager::ICONS,
        'fa4compatibility' => 'overlay_selected_icon',
        'skin'             => 'inline',
        'label_block'      => false,
      ]
    );
    $this->add_control(
      'overlay_title',
      [
        'label' => esc_html__('Title', 'tjcore'),
        'type' => Controls_Manager::SELECT,
        'default' => '',
        'options' => [
          '' => esc_html__('None', 'tjcore'),
          'title' => esc_html__('Title', 'tjcore'),
          'caption' => esc_html__('Caption', 'tjcore'),
          'alt' => esc_html__('Alt', 'tjcore'),
          'description' => esc_html__('Description', 'tjcore'),
        ],
        'frontend_available' => true,
      ]
    );
    $this->add_control(
      'overlay_description',
      [
        'label' => esc_html__('Description', 'tjcore'),
        'type' => Controls_Manager::SELECT,
        'default' => '',
        'options' => [
          '' => esc_html__('None', 'tjcore'),
          'title' => esc_html__('Title', 'tjcore'),
          'caption' => esc_html__('Caption', 'tjcore'),
          'alt' => esc_html__('Alt', 'tjcore'),
          'description' => esc_html__('Description', 'tjcore'),
        ],
        'frontend_available' => true,
      ]
    );
    $this->end_controls_section();

    /** ---------------------------------------------------------------
     * TAB_STYLE
     * -------------------------------------------------------------* */

    // overlay_background
    $this->start_controls_section(
      'image_style',
      [
        'label' => esc_html__('Image', 'tjcore'),
        'tab' => Controls_Manager::TAB_STYLE,
      ]
    );
    $this->start_controls_tabs('image_tabs');
    $this->start_controls_tab(
      'image_normal',
      [
        'label' => esc_html__('Normal', 'tjcore'),
      ]
    );
    $this->add_control(
      'image_border_color',
      [
        'label' => esc_html__('Border Color', 'tjcore'),
        'type' => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}}' => '--image-border-color: {{VALUE}}',
        ],
      ]
    );
    $this->add_control(
      'image_border_width',
      [
        'label' => esc_html__('Border Width', 'tjcore'),
        'type' => Controls_Manager::SLIDER,
        'size_units' => ['px', '%', 'em', 'rem', 'vw', 'custom'],
        'range' => [
          'px' => [
            'max' => 20,
          ],
          'em' => [
            'max' => 2,
          ],
          'em' => [
            'max' => 2,
          ],
        ],
        'selectors' => [
          '{{WRAPPER}}' => '--image-border-width: {{SIZE}}{{UNIT}};',
        ],
      ]
    );
    $this->add_control(
      'image_border_radius',
      [
        'label' => esc_html__('Border Radius', 'tjcore'),
        'type' => Controls_Manager::SLIDER,
        'size_units' => ['px', '%', 'em', 'rem', 'custom'],
        'selectors' => [
          '{{WRAPPER}}' => '--image-border-radius: {{SIZE}}{{UNIT}};',
        ],
      ]
    );
    $this->add_group_control(
      Group_Control_Css_Filter::get_type(),
      [
        'name' => 'image_css_filters',
        'selector' => '{{WRAPPER}} .e-gallery-image',
      ]
    );
    $this->end_controls_tab();
    $this->start_controls_tab(
      'image_hover',
      [
        'label' => esc_html__('Hover', 'tjcore'),
      ]
    );
    $this->add_control(
      'image_border_color_hover',
      [
        'label' => esc_html__('Border Color', 'tjcore'),
        'type' => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .elementor-gallery-item:hover' => 'border-color: {{VALUE}}',
        ],
      ]
    );
    $this->add_control(
      'image_border_radius_hover',
      [
        'label' => esc_html__('Border Radius', 'tjcore'),
        'type' => Controls_Manager::SLIDER,
        'size_units' => ['px', '%', 'em', 'rem', 'custom'],
        'selectors' => [
          '{{WRAPPER}} .elementor-gallery-item:hover' => 'border-radius: {{SIZE}}{{UNIT}};',
        ],
      ]
    );
    $this->add_group_control(
      Group_Control_Css_Filter::get_type(),
      [
        'name' => 'image_css_filters_hover',
        'selector' => '{{WRAPPER}} .e-gallery-item:hover .e-gallery-image',
      ]
    );
    $this->end_controls_tab(); // overlay_background normal
    $this->end_controls_tabs(); // overlay_background tabs

    $this->add_control(
      'image_hover_animation',
      [
        'label' => esc_html__('Hover Animation', 'tjcore'),
        'type' => Controls_Manager::SELECT,
        'options' => [
          '' => 'None',
          'grow' => 'Zoom In',
          'shrink-contained' => 'Zoom Out',
          'move-contained-left' => 'Move Left',
          'move-contained-right' => 'Move Right',
          'move-contained-top' => 'Move Up',
          'move-contained-bottom' => 'Move Down',
        ],
        'separator' => 'before',
        'default' => '',
        'frontend_available' => true,
        'render_type' => 'ui',
      ]
    );
    $this->add_control(
      'image_animation_duration',
      [
        'label' => esc_html__('Animation Duration', 'tjcore') . ' (ms)',
        'type' => Controls_Manager::SLIDER,
        'default' => [
          'size' => 800,
        ],
        'range' => [
          'px' => [
            'min' => 0,
            'max' => 3000,
            'step' => 100,
          ],
        ],
        'selectors' => [
          '{{WRAPPER}}' => '--image-transition-duration: {{SIZE}}ms',
        ],
      ]
    );
    $this->end_controls_section();

    // background overlay
    $this->start_controls_section(
      'overlay_style',
      [
        'label' => esc_html__('Overlay', 'tjcore'),
        'tab' => Controls_Manager::TAB_STYLE,
        'condition' => [
          'overlay_background' => 'yes',
        ],
      ]
    );
    $this->start_controls_tabs('overlay_background_tabs');
    $this->start_controls_tab(
      'overlay_normal',
      [
        'label' => esc_html__('Normal', 'tjcore'),
      ]
    );
    $this->add_group_control(
      Group_Control_Background::get_type(),
      [
        'name' => 'overlay_background',
        'types' => ['classic', 'gradient'],
        'exclude' => ['image'],
        'selector' => '{{WRAPPER}} .elementor-gallery-item__overlay',
        'fields_options' => [
          'background' => [
            'label' => esc_html__('Overlay', 'tjcore'),
          ],
        ],
      ]
    );
    $this->end_controls_tab(); // overlay_background normal
    $this->start_controls_tab(
      'overlay_hover',
      [
        'label' => esc_html__('Hover', 'tjcore'),
      ]
    );
    $this->add_group_control(
      Group_Control_Background::get_type(),
      [
        'name' => 'overlay_background_hover',
        'types' => ['classic', 'gradient'],
        'selector' => '{{WRAPPER}} .e-gallery-item:hover .elementor-gallery-item__overlay, {{WRAPPER}} .e-gallery-item:focus .elementor-gallery-item__overlay',
        'exclude' => ['image'],
        'fields_options' => [
          'background' => [
            'default' => 'classic',
          ],
          'color' => [
            'default' => 'rgba(0,0,0,0.5)',
          ],
        ],
      ]
    );
    $this->end_controls_tab(); // overlay_background normal
    $this->end_controls_tabs(); // overlay_background tabs
    $this->add_control(
      'image_blend_mode',
      [
        'label' => esc_html__('Blend Mode', 'tjcore'),
        'type' => Controls_Manager::SELECT,
        'default' => '',
        'options' => [
          '' => esc_html__('Normal', 'tjcore'),
          'multiply' => 'Multiply',
          'screen' => 'Screen',
          'overlay' => 'Overlay',
          'darken' => 'Darken',
          'lighten' => 'Lighten',
          'color-dodge' => 'Color Dodge',
          'color-burn' => 'Color Burn',
          'hue' => 'Hue',
          'saturation' => 'Saturation',
          'color' => 'Color',
          'exclusion' => 'Exclusion',
          'luminosity' => 'Luminosity',
        ],
        'selectors' => [
          '{{WRAPPER}}' => '--overlay-mix-blend-mode: {{VALUE}}',
        ],
        'separator' => 'before',
        'render_type' => 'ui',
      ]
    );
    $this->add_control(
      'background_overlay_hover_animation',
      [
        'label' => esc_html__('Hover Animation', 'tjcore'),
        'type' => Controls_Manager::SELECT,
        'groups' => [
          [
            'label' => esc_html__('None', 'tjcore'),
            'options' => [
              '' => esc_html__('None', 'tjcore'),
            ],
          ],
          [
            'label' => esc_html__('Entrance', 'tjcore'),
            'options' => [
              'enter-from-right' => 'Slide In Right',
              'enter-from-left' => 'Slide In Left',
              'enter-from-top' => 'Slide In Up',
              'enter-from-bottom' => 'Slide In Down',
              'enter-zoom-in' => 'Zoom In',
              'enter-zoom-out' => 'Zoom Out',
              'fade-in' => 'Fade In',
            ],
          ],
          [
            'label' => esc_html__('Exit', 'tjcore'),
            'options' => [
              'exit-to-right' => 'Slide Out Right',
              'exit-to-left' => 'Slide Out Left',
              'exit-to-top' => 'Slide Out Up',
              'exit-to-bottom' => 'Slide Out Down',
              'exit-zoom-in' => 'Zoom In',
              'exit-zoom-out' => 'Zoom Out',
              'fade-out' => 'Fade Out',
            ],
          ],
        ],
        'separator' => 'before',
        'default' => '',
        'frontend_available' => true,
        'render_type' => 'ui',
      ]
    );
    $this->add_control(
      'background_overlay_animation_duration',
      [
        'label' => esc_html__('Animation Duration', 'tjcore') . ' (ms)',
        'type' => Controls_Manager::SLIDER,
        'default' => [
          'size' => 800,
        ],
        'range' => [
          'px' => [
            'min' => 0,
            'max' => 3000,
            'step' => 100,
          ],
        ],
        'selectors' => [
          '{{WRAPPER}}' => '--overlay-transition-duration: {{SIZE}}ms',
        ],
      ]
    );
    $this->end_controls_section();

    // overlay content
    $this->start_controls_section(
      'overlay_content_style',
      [
        'label' => esc_html__('Content', 'tjcore'),
        'tab' => Controls_Manager::TAB_STYLE,
        //TODO: add conditions for this section
      ]
    );
    $this->add_control(
      'content_alignment',
      [
        'label' => esc_html__('Alignment', 'tjcore'),
        'type' => Controls_Manager::CHOOSE,
        'options' => [
          'start' => [
            'title' => esc_html__('Left', 'tjcore'),
            'icon' => 'eicon-text-align-left',
          ],
          'center' => [
            'title' => esc_html__('Center', 'tjcore'),
            'icon' => 'eicon-text-align-center',
          ],
          'end' => [
            'title' => esc_html__('Right', 'tjcore'),
            'icon' => 'eicon-text-align-right',
          ],
        ],
        'default' => 'center',
        'selectors' => [
          '{{WRAPPER}}' => '--content-text-align: {{VALUE}}',
        ],
      ]
    );
    $this->add_control(
      'content_vertical_position',
      [
        'label' => esc_html__('Vertical Position', 'tjcore'),
        'type' => Controls_Manager::CHOOSE,
        'options' => [
          'top' => [
            'title' => esc_html__('Top', 'tjcore'),
            'icon' => 'eicon-v-align-top',
          ],
          'middle' => [
            'title' => esc_html__('Middle', 'tjcore'),
            'icon' => 'eicon-v-align-middle',
          ],
          'bottom' => [
            'title' => esc_html__('Bottom', 'tjcore'),
            'icon' => 'eicon-v-align-bottom',
          ],
        ],
        'selectors_dictionary' => [
          'top' => 'flex-start',
          'middle' => 'center',
          'bottom' => 'flex-end',
        ],
        'selectors' => [
          '{{WRAPPER}}' => '--content-justify-content: {{VALUE}}',
        ],
      ]
    );
    $this->add_responsive_control(
      'content_padding',
      [
        'label' => esc_html__('Padding', 'tjcore'),
        'type' => Controls_Manager::SLIDER,
        'size_units' => ['px', '%', 'em', 'rem', 'vw', 'custom'],
        'default' => [
          'size' => 20,
        ],
        'selectors' => [
          '{{WRAPPER}}' => '--content-padding: {{SIZE}}{{UNIT}}',
        ],
      ]
    );
    $this->add_control(
      'icon_title',
      [
        'label' => esc_html__('Icon', 'tjcore'),
        'type' => Controls_Manager::HEADING,
        'separator' => 'before',
        'condition' => [
          'overlay_icon[value]!' => '',
        ],
      ]
    );
    $this->add_responsive_control(
      'icon_width',
      [
        'label' => esc_html__('Width', 'tjcore'),
        'type' => Controls_Manager::SLIDER,
        'size_units' => ['px', '%', 'em', 'rem', 'vw', 'custom'],
        'selectors' => [
          '{{WRAPPER}} .elementor-gallery-item__icon' => 'width: {{SIZE}}{{UNIT}}',
        ],
        'condition' => [
          'overlay_icon[value]!' => '',
        ],
      ]
    );
    $this->add_responsive_control(
      'icon_height',
      [
        'label' => esc_html__('Height', 'tjcore'),
        'type' => Controls_Manager::SLIDER,
        'size_units' => ['px', '%', 'em', 'rem', 'vw', 'custom'],
        'selectors' => [
          '{{WRAPPER}} .elementor-gallery-item__icon' => 'height: {{SIZE}}{{UNIT}}',
        ],
        'condition' => [
          'overlay_icon[value]!' => '',
        ],
      ]
    );
    $this->add_responsive_control(
      'icon_size',
      [
        'label' => esc_html__('Icon Size', 'tjcore'),
        'type' => Controls_Manager::SLIDER,
        'size_units' => ['px', '%', 'em', 'rem', 'vw', 'custom'],
        'selectors' => [
          '{{WRAPPER}} .elementor-gallery-item__icon' => 'font-size: {{SIZE}}{{UNIT}}',
          '{{WRAPPER}} .elementor-gallery-item__icon svg' => 'max-width: {{SIZE}}{{UNIT}}',
        ],
        'condition' => [
          'overlay_icon[value]!' => '',
        ],
      ]
    );
    $this->add_control(
      'icon_bg_color',
      [
        'label' => esc_html__('Background Color', 'tjcore'),
        'type' => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .elementor-gallery-item__icon' => 'background-color: {{VALUE}}',
        ],
        'condition' => [
          'overlay_icon[value]!' => '',
        ],
      ]
    );
    $this->add_responsive_control(
      'icon_backdrop_filter',
      [
        'label' => esc_html__('Backdrop Filter', 'tjcore'),
        'type' => Controls_Manager::SLIDER,
        'size_units' => ['px'],
        'selectors' => [
          '{{WRAPPER}} .elementor-gallery-item__icon' => 'blur({{SIZE}}{{UNIT}})',
        ],
        'condition' => [
          'overlay_icon[value]!' => '',
        ],
      ]
    );
    $this->add_control(
      'icon_color',
      [
        'label' => esc_html__('Icon Color', 'tjcore'),
        'type' => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .elementor-gallery-item__icon' => 'color: {{VALUE}}',
        ],
        'condition' => [
          'overlay_icon[value]!' => '',
        ],
      ]
    );
    $this->add_control(
      'icon_spacing',
      [
        'label' => esc_html__('Icon Spacing', 'tjcore'),
        'type' => Controls_Manager::SLIDER,
        'size_units' => ['px', '%', 'em', 'rem', 'custom'],
        'selectors' => [
          '{{WRAPPER}}' => '--title-margin-top: {{SIZE}}{{UNIT}}',
        ],
        'condition' => [
          'overlay_icon[value]!' => '',
        ],
      ]
    );
    $this->add_control(
      'heading_title',
      [
        'label' => esc_html__('Title', 'tjcore'),
        'type' => Controls_Manager::HEADING,
        'separator' => 'before',
        'condition' => [
          'overlay_title!' => '',
        ],
      ]
    );
    $this->add_control(
      'title_color',
      [
        'label' => esc_html__('Color', 'tjcore'),
        'type' => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}}' => '--title-text-color: {{VALUE}}',
        ],
        'condition' => [
          'overlay_title!' => '',
        ],
      ]
    );
    $this->add_group_control(
      Group_Control_Typography::get_type(),
      [
        'name' => 'title_typography',
        'global' => [
          'default' => Global_Typography::TYPOGRAPHY_PRIMARY,
        ],
        'selector' => '{{WRAPPER}} .elementor-gallery-item__title',
        'condition' => [
          'overlay_title!' => '',
        ],
      ]
    );
    $this->add_control(
      'title_spacing',
      [
        'label' => esc_html__('Spacing', 'tjcore'),
        'type' => Controls_Manager::SLIDER,
        'size_units' => ['px', '%', 'em', 'rem', 'custom'],
        'selectors' => [
          '{{WRAPPER}}' => '--description-margin-top: {{SIZE}}{{UNIT}}',
        ],
        'condition' => [
          'overlay_title!' => '',
        ],
      ]
    );
    $this->add_control(
      'heading_description',
      [
        'label' => esc_html__('Description', 'tjcore'),
        'type' => Controls_Manager::HEADING,
        'separator' => 'before',
        'condition' => [
          'overlay_description!' => '',
        ],
      ]
    );
    $this->add_control(
      'description_color',
      [
        'label' => esc_html__('Color', 'tjcore'),
        'type' => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}}' => '--description-text-color: {{VALUE}}',
        ],
        'condition' => [
          'overlay_description!' => '',
        ],
      ]
    );
    $this->add_group_control(
      Group_Control_Typography::get_type(),
      [
        'name' => 'description_typography',
        'global' => [
          'default' => Global_Typography::TYPOGRAPHY_TEXT,
        ],
        'selector' => '{{WRAPPER}} .elementor-gallery-item__description',
        'condition' => [
          'overlay_description!' => '',
        ],
      ]
    );
    $this->add_control(
      'content_hover_animation',
      [
        'label' => esc_html__('Hover Animation', 'tjcore'),
        'type' => Controls_Manager::SELECT,
        'groups' => [
          [
            'label' => esc_html__('None', 'tjcore'),
            'options' => [
              '' => esc_html__('None', 'tjcore'),
            ],
          ],
          [
            'label' => esc_html__('Entrance', 'tjcore'),
            'options' => [
              'enter-from-right' => 'Slide In Right',
              'enter-from-left' => 'Slide In Left',
              'enter-from-top' => 'Slide In Up',
              'enter-from-bottom' => 'Slide In Down',
              'enter-zoom-in' => 'Zoom In',
              'enter-zoom-out' => 'Zoom Out',
              'fade-in' => 'Fade In',
            ],
          ],
          [
            'label' => esc_html__('Reaction', 'tjcore'),
            'options' => [
              'grow' => 'Grow',
              'shrink' => 'Shrink',
              'move-right' => 'Move Right',
              'move-left' => 'Move Left',
              'move-up' => 'Move Up',
              'move-down' => 'Move Down',
            ],
          ],
          [
            'label' => esc_html__('Exit', 'tjcore'),
            'options' => [
              'exit-to-right' => 'Slide Out Right',
              'exit-to-left' => 'Slide Out Left',
              'exit-to-top' => 'Slide Out Up',
              'exit-to-bottom' => 'Slide Out Down',
              'exit-zoom-in' => 'Zoom In',
              'exit-zoom-out' => 'Zoom Out',
              'fade-out' => 'Fade Out',
            ],
          ],
        ],
        'default' => 'fade-in',
        'separator' => 'before',
        'render_type' => 'ui',
        'frontend_available' => true,
      ]
    );
    $this->add_control(
      'content_animation_duration',
      [
        'label' => esc_html__('Animation Duration', 'tjcore') . ' (ms)',
        'type' => Controls_Manager::SLIDER,
        'default' => [
          'size' => 800,
        ],
        'range' => [
          'px' => [
            'min' => 0,
            'max' => 3000,
            'step' => 100,
          ],
        ],
        'selectors' => [
          '{{WRAPPER}}' => '--content-transition-duration: {{SIZE}}ms; --content-transition-delay: {{SIZE}}ms;',
        ],
        'condition' => [
          'content_hover_animation!' => '',
        ],
      ]
    );
    $this->add_control(
      'content_sequenced_animation',
      [
        'label' => esc_html__('Sequenced Animation', 'tjcore'),
        'type' => Controls_Manager::SWITCHER,
        'condition' => [
          'content_hover_animation!' => '',
        ],
        'frontend_available' => true,
        'render_type' => 'ui',
      ]
    );
    $this->end_controls_section();
  }

  /**
   * the widget output on the frontend.
   */
  protected function render_static() {

    $settings = $this->get_settings_for_display();

    $gap = $settings['gap']['size'] . $settings['gap']['unit'];
    $ratio_percentage = '75';
    $columns = 4;

    if ($settings['columns']) {
      $columns = $settings['columns'];
    }

    if ($settings['aspect_ratio']) {
      $ratio_array = explode(':', $settings['aspect_ratio']);

      $ratio_percentage = ($ratio_array[1] / $ratio_array[0]) * 100;
    }

    $this->add_render_attribute(
      'gallery_container',
      [
        'style' => "--columns: {$columns}; --aspect-ratio: {$ratio_percentage}%; --hgap: {$gap}; --vgap: {$gap};",
        'class' => 'e-gallery-grid',
      ]
    );

    $galleries = [];

    $galleries[0] = $settings['gallery'];

    foreach ($galleries as $gallery) {
      foreach ($gallery as $item) {
        $image_src = 'custom' !== $settings['thumbnail_image_size']
          ? wp_get_attachment_image_src($item['id'], $settings['thumbnail_image_size'])[0]
          : Group_Control_Image_Size::get_attachment_image_src($item['id'], 'thumbnail_image', $settings);

        $this->add_render_attribute('gallery_item_image_' . $item['id'], [
          'style' => "background-image: url('{$image_src}');",
        ]);
      }
    }

    $this->render();
  }

  /**
   * Render
   */
  protected function render() {
    $settings = $this->get_settings_for_display();

    $has_icon = ! empty($settings['overlay_icon']);

    $has_description = ! empty($settings['overlay_description']);

    $has_title = ! empty($settings['overlay_title']);

    $has_animation = ! empty($settings['image_hover_animation']) || ! empty($settings['content_hover_animation']) || ! empty($settings['background_overlay_hover_animation']);

    $gallery_item_tag = ! empty($settings['link_to']) ? 'a' : 'div';

    $galleries = [];

    if (!empty($settings['gallery'])) {
      $galleries[0] = $settings['gallery'];
    } elseif (Plugin::$instance->editor->is_edit_mode()) { ?>
      <i class="elementor-widget-empty-icon eicon-gallery-justified"></i>
    <?php }

    $this->add_render_attribute('gallery_container', 'class', 'elementor-gallery__container');

    if ($has_icon || $has_title || $has_description) {
      $this->add_render_attribute('gallery_item_content', 'class', 'elementor-gallery-item__content');

      if ($has_icon) {
        $this->add_render_attribute('gallery_item_icon', 'class', 'elementor-gallery-item__icon');
      }

      if ($has_title) {
        $this->add_render_attribute('gallery_item_title', 'class', 'elementor-gallery-item__title');
      }

      if ($has_description) {
        $this->add_render_attribute('gallery_item_description', 'class', 'elementor-gallery-item__description');
      }
    }

    $this->add_render_attribute('gallery_item_background_overlay', ['class' => 'elementor-gallery-item__overlay']);

    $gallery_items = [];
    $thumbnail_size = $settings['thumbnail_image_size'];
    foreach ($galleries as $gallery_index => $gallery) {
      foreach ($gallery as $index => $item) {
        if (in_array($item['id'], array_keys($gallery_items), true)) {
          $gallery_items[$item['id']][] = $gallery_index;
        } else {
          $gallery_items[$item['id']] = [$gallery_index];
        }
      }
    }

    if ('random' === $settings['order_by']) {
      $shuffled_items = [];
      $keys = array_keys($gallery_items);
      shuffle($keys);
      foreach ($keys as $key) {
        $shuffled_items[$key] = $gallery_items[$key];
      }
      $gallery_items = $shuffled_items;
    }

    if (! empty($galleries)) { ?>
      <div <?php $this->print_render_attribute_string('gallery_container'); ?>>
        <?php
        foreach ($gallery_items as $id => $tags) :
          $unique_index = $id; //$gallery_index . '_' . $index;
          $image_src = wp_get_attachment_image_src($id, $thumbnail_size);
          if (! $image_src) {
            continue;
          }
          $attachment = get_post($id);

          $image_data = $this->get_image_data($attachment, $id, $image_src, $settings);

          $this->add_render_attribute('gallery_item_' . $unique_index, [
            'class' => [
              'e-gallery-item',
              'elementor-gallery-item',
            ],
          ]);

          if ($has_animation) {
            $this->add_render_attribute('gallery_item_' . $unique_index, ['class' => 'elementor-animated-content']);
          }

          // if ($is_multiple) {
          //   $this->add_render_attribute('gallery_item_' . $unique_index, ['data-e-gallery-tags' => implode(',', $tags)]);
          // }

          if ($has_title && 'div' === $gallery_item_tag) {
            $this->add_render_attribute('gallery_item_' . $unique_index, ['tabindex' => '0']);
          }

          if ('a' === $gallery_item_tag) {
            if ('file' === $settings['link_to']) {
              $href = $image_data['media'];

              $this->add_render_attribute('gallery_item_' . $unique_index, [
                'href' => esc_url($href),
              ]);

              if (Plugin::$instance->editor->is_edit_mode()) {
                $this->add_render_attribute('gallery_item_' . $unique_index, 'class', 'elementor-clickable');
              }

              $this->add_lightbox_data_attributes('gallery_item_' . $unique_index, $id, $settings['open_lightbox'], $this->get_id());
            } elseif ('custom' === $settings['link_to']) {
              $this->add_link_attributes('gallery_item_' . $unique_index, $settings['url']);
            }
          }

          $this->add_render_attribute(
            'gallery_item_image_' . $unique_index,
            [
              'class' => [
                'e-gallery-image',
                'elementor-gallery-item__image',
              ],
              'data-thumbnail' => $image_data['src'],
              'data-width' => $image_data['width'],
              'data-height' => $image_data['height'],
              'aria-label' => $image_data['alt'],
              'role' => 'img',
            ]
          ); ?>


          <<?php Utils::print_validated_html_tag($gallery_item_tag); ?> <?php $this->print_render_attribute_string('gallery_item_' . $unique_index); ?>>
            <div <?php $this->print_render_attribute_string('gallery_item_image_' . $unique_index); ?>></div>
            <?php if (! empty($settings['overlay_background'])) : ?>
              <div <?php $this->print_render_attribute_string('gallery_item_background_overlay'); ?>></div>
            <?php endif; ?>

            <?php if ($has_icon || $has_title || $has_description) : ?>
              <div <?php $this->print_render_attribute_string('gallery_item_content'); ?>>
                <?php if ($has_icon) :
                  $icon = $settings['overlay_icon'];
                  if (!empty($icon['value'])): ?>
                    <div <?php $this->print_render_attribute_string('gallery_item_icon'); ?>>
                      <?php \Elementor\Icons_Manager::render_icon($icon, ['aria-hidden' => 'true']); ?>
                    </div>
                    <?php endif;
                  if ($has_title) :
                    $title = $image_data[$settings['overlay_title']];

                    if (! empty($title)) : ?>
                      <div <?php $this->print_render_attribute_string('gallery_item_title'); ?>>
                        <?php // PHPCS - the main text of a widget should not be escaped. 
                        ?>
                        <?php echo $title; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped 
                        ?>
                      </div>
                    <?php endif;
                  endif;
                endif;
                if ($has_description) :
                  $description = $image_data[$settings['overlay_description']];
                  if (! empty($description)) : ?>
                    <div <?php $this->print_render_attribute_string('gallery_item_description'); ?>>
                      <?php // PHPCS - the main text of a widget should not be escaped. 
                      ?>
                      <?php echo $description; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped 
                      ?>
                    </div>
                <?php endif;
                endif; ?>
              </div>
            <?php endif; ?>
          </<?php Utils::print_validated_html_tag($gallery_item_tag); ?>>
        <?php endforeach;
        //endforeach; 
        ?>
      </div>
<?php }
  }

  protected function get_image_data($attachment, $image_id, $image_src, $settings): array {
    $image_data = [
      'alt' => get_post_meta($attachment->ID, '_wp_attachment_image_alt', true),
      'media' => wp_get_attachment_image_src($image_id, 'full')['0'],
      'src' => $image_src['0'],
      'width' => $image_src['1'],
      'height' => $image_src['2'],
      'caption' => $attachment->post_excerpt,
      'description' => $attachment->post_content,
      'title' => $attachment->post_title,
    ];

    if ('custom' !== $settings['thumbnail_image_size']) {
      return $image_data;
    }

    $image_data['src'] = Group_Control_Image_Size::get_attachment_image_src($image_id, 'thumbnail_image', $settings);
    $image_data['width'] = $settings['thumbnail_image_custom_dimension']['width'];
    $image_data['height'] = $settings['thumbnail_image_custom_dimension']['height'];

    return $image_data;
  }
}
$widgets_manager->register(new TJ_Gallery());
