<?php

namespace App\Http\Controllers\Api\Panel;

use App\Http\Controllers\Api\Controller;
use App\Http\Resources\CertificateResource;
use App\Mixins\Certificate\MakeCertificate;
use App\Models\Api\Certificate;
use App\Models\CertificateTemplate;
use App\Models\Api\Quiz;
use App\Models\Api\QuizzesResult;
use App\Models\Reward;
use App\Models\RewardAccounting;
use Illuminate\Support\Facades\Validator;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Intervention\Image\Facades\Image;

class CertificatesController extends Controller
{
    public function created(Request $request)
    {
        $user = apiAuth();

        $quizzes = Quiz::where('creator_id', $user->id)
            ->where('status', Quiz::ACTIVE)->handleFilters()->get();

        return apiResponse2(1, 'retrieved', trans('public.retrieved'), [
            'certificates' => CertificateResource::collection($quizzes),
        ]);


    }

    public function students()
    {
        $user = apiAuth();

        $quizzes = Quiz::where('creator_id', $user->id)
            ->pluck('id')->toArray();


        $ee = Certificate::whereIn('quiz_id', $quizzes)
            ->get()
            ->map(function ($certificate) {

                return $certificate->details;

            });

        return apiResponse2(1, 'retrieved', trans('public.retrieved'), $ee);
    }

    public function achievements(Request $request)
    {
        $user = apiAuth();
        
        // First try to get certificates from the certifications table using id_no matching student_id
        $certifications = \DB::table('certifications')
            ->where('id_no', $user->student_id)
            ->get();
            
        if ($certifications->isNotEmpty()) {
            $certificatesData = $certifications->map(function ($certification) {
                return [
                    'id' => $certification->id,
                    'student_id' => $certification->id_no,
                    'type' => 'certification',
                    'status' => 'generated',
                    'course_title' => $certification->course_name,
                    'student_name' => $certification->name,
                    'issued_date' => $certification->issue_date,
                    'expiration_date' => $certification->expiration_date,
                    'company' => $certification->company,
                    'certificate_number' => $certification->neft_certificate_number,
                    'offline_data' => [
                        'certificate_id' => $certification->id,
                        'student_name' => $certification->name,
                        'course_title' => $certification->course_name,
                        'grade' => 'N/A',
                        'issued_date' => $certification->issue_date,
                        'expiration_date' => $certification->expiration_date,
                        'company' => $certification->company,
                        'certificate_number' => $certification->neft_certificate_number,
                        'verification_code' => $certification->neft_certificate_number,
                        'template_data' => []
                    ]
                ];
            });
            
            return apiResponse2(1, 'retrieved', trans('public.retrieved'), $certificatesData);
        }
        
        // Fallback to existing certificates table
        $certificates = Certificate::where('student_id', $user->id)
            ->where('type', 'quiz')
            ->where('status', Certificate::STATUS_GENERATED)
            ->with(['quiz', 'student', 'webinar'])
            ->get();
            
        if ($certificates->isNotEmpty()) {
            $certificatesData = $certificates->map(function ($certificate) {
                return [
                    'id' => $certificate->id,
                    'student_id' => $certificate->student_id,
                    'quiz_id' => $certificate->quiz_id,
                    'webinar_id' => $certificate->webinar_id,
                    'type' => $certificate->type,
                    'status' => $certificate->status,
                    'user_grade' => $certificate->user_grade,
                    'image_path' => $certificate->image_path,
                    'html_content' => $certificate->html_content,
                    'qr_code' => $certificate->qr_code,
                    'verification_url' => $certificate->verification_url,
                    'created_at' => $certificate->created_at,
                    'generated_at' => $certificate->generated_at,
                    'metadata' => $certificate->metadata,
                    'offline_data' => [
                        'certificate_id' => $certificate->id,
                        'student_name' => $certificate->student->full_name ?? '',
                        'course_title' => $certificate->quiz->webinar->title ?? $certificate->quiz->title ?? '',
                        'grade' => $certificate->user_grade ?? '',
                        'issued_date' => $certificate->created_at ? date('F j, Y', $certificate->created_at) : '',
                        'verification_code' => 'CERT' . $certificate->id,
                        'template_data' => []
                    ]
                ];
            });
            
            return apiResponse2(1, 'retrieved', trans('public.retrieved'), $certificatesData);
        }
        
        // Fallback to quiz results if no certificates found
        $results = QuizzesResult::where('user_id', $user->id)->where('status', QuizzesResult::$passed)
            ->whereHas('quiz', function ($query) {
                $query->where('status', Quiz::ACTIVE);
            })
            ->get()->map(function ($result) {

                return array_merge($result->details,
                    ['certificate' => $result->certificate->brief ?? null]
                );

            });

        return apiResponse2(1, 'retrieved', trans('public.retrieved'), $results);
    }

    public function makeCertificate($quizResultId)
    {
        $user = apiAuth();

        $makeCertificate = new MakeCertificate();

        $quizResult = QuizzesResult::where('id', $quizResultId)
            ->where('user_id', $user->id)
            ->where('status', QuizzesResult::$passed)
            ->with(['quiz' => function ($query) {
                $query->with(['webinar']);
            }])
            ->first();

        if (!empty($quizResult)) {
            return $makeCertificate->makeQuizCertificate($quizResult);
        }

        abort(404);
    }

    /**
     * Get certificate HTML content for mobile app
     */
    public function getCertificateHtml($certificateId)
    {
        $user = apiAuth();
        
        // Find the certificate
        $certificate = Certificate::where('id', $certificateId)
            ->where('student_id', $user->id)
            ->first();
            
        if (!$certificate) {
            return response()->json(['error' => 'Certificate not found'], 404);
        }
        
        // Get the appropriate template based on certificate type
        $template = CertificateTemplate::where('status', 'publish')
            ->where('type', $certificate->type)
            ->first();
            
        if (!$template) {
            return response()->json(['error' => 'Certificate template not found'], 404);
        }
        
        // Generate the certificate body
        $makeCertificate = new MakeCertificate();
        
        $body = '';
        $courseTitle = '';
        $teacherName = '';
        $duration = '';
        
        if ($certificate->type === 'quiz' && $certificate->quiz) {
            $quiz = $certificate->quiz;
            $body = $makeCertificate->makeBody(
                $template,
                $certificate,
                $user,
                $template->body,
                $quiz->webinar ? $quiz->webinar->title : '-',
                $certificate->user_grade,
                $quiz->webinar ? $quiz->webinar->teacher->id : null,
                $quiz->webinar ? $quiz->webinar->teacher->full_name : null,
                $quiz->webinar ? $quiz->webinar->duration : null
            );
        } elseif ($certificate->type === 'course' && $certificate->webinar) {
            $course = $certificate->webinar;
            $locale = app()->getLocale();
            $templateBody = (!empty($template->translate($locale)) and !empty($template->translate($locale)->body)) 
                ? $template->translate($locale)->body 
                : $template->body;
                
            $body = $makeCertificate->makeBody(
                $template,
                $certificate,
                $user,
                $templateBody,
                $course->title,
                null,
                $course->teacher->id,
                $course->teacher->full_name,
                $course->duration
            );
        } elseif ($certificate->type === 'bundle' && $certificate->bundle) {
            $bundle = $certificate->bundle;
            $locale = app()->getLocale();
            $templateBody = (!empty($template->translate($locale)) and !empty($template->translate($locale)->body)) 
                ? $template->translate($locale)->body 
                : $template->body;
                
            $body = $makeCertificate->makeBody(
                $template,
                $certificate,
                $user,
                $templateBody,
                $bundle->title,
                null,
                $bundle->teacher->id,
                $bundle->teacher->full_name,
                $bundle->duration
            );
        }
        
        $data = [
            'body' => $body
        ];
        
        // Generate the HTML using the same template as the backend
        $html = (string)view()->make('admin.certificates.create_template.show_certificate', $data);
        
        return response($html, 200, [
            'Content-Type' => 'text/html; charset=utf-8'
        ]);
    }

}

