<?php

namespace App\Http\Controllers\Api\Auth;

use App\Http\Controllers\Api\Auth\VerificationController;
use App\Http\Controllers\Api\Controller;
use App\Http\Controllers\Web\traits\UserFormFieldsTrait;
use App\Mixins\RegistrationBonus\RegistrationBonusAccounting;
use App\Models\Affiliate;
use App\Models\Reward;
use App\Models\RewardAccounting;
use App\Models\Role;
use App\Models\UserFormField;
use App\Models\UserMeta;
use App\User;
use Illuminate\Auth\Events\Registered;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class RegisterController extends Controller
{
    use UserFormFieldsTrait;
    /*
    |--------------------------------------------------------------------------
    | Register Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles the registration of new users as well as their
    | validation and creation. By default this controller uses a trait to
    | provide this functionality without requiring any additional code.
    |
    */
    public function stepRegister(Request $request, $step)
    {
        if ($step == 1) {
            return $this->stepOne($request);

        } elseif ($step == 2) {
            return $this->stepTwo($request);

        } elseif ($step == 3) {
            return $this->stepThree($request);
        }
        abort(404);

    }

    private function stepOne(Request $request)
    {
        $registerMethod = getGeneralSettings('register_method') ?? 'mobile';
        $data = $request->all();
        $username = $this->username();

        if ($registerMethod !== $username && $username) {
            return apiResponse2(0, 'invalid_register_method', trans('api.auth.invalid_register_method'));
        }

        $rules = [
            'country_code' => ($username == 'mobile') ? 'required' : 'nullable',
            // Ensure email/mobile is unique to prevent duplicate accounts
            $username => ($username == 'mobile') ? 'required|numeric|unique:users,mobile' : 'required|string|email|max:255|unique:users,email',
            'password' => 'required|string|min:6|confirmed',
            'password_confirmation' => 'required|same:password',
            // Made optional per Apple's data collection policies
            'student_id' => 'nullable|numeric|digits_between:8,15|unique:users,student_id',
        ];

        validateParam($data, $rules);

        if ($username == 'mobile') {
            $data[$username] = ltrim($data['country_code'], '+') . ltrim($data[$username], '0');
        }

        $verificationValue = ($registerMethod == "mobile") ? "+{$data[$username]}" : $data[$username];

        // Check if user already exists
        $userCase = User::where($username, $data[$username])->first();

        if ($userCase) {
            // User already exists - prevent registration to protect existing accounts
            return apiResponse2(0, 'already_registered', trans('api.auth.user_already_exists_please_login'), [
                'message' => trans('api.auth.user_already_exists_please_login')
            ]);
        }


        $referralSettings = getReferralSettings();
        $usersAffiliateStatus = (!empty($referralSettings) and !empty($referralSettings['users_affiliate_status']));


        $disableRegistrationVerificationProcess = getGeneralOptionsSettings('disable_registration_verification_process');

        // Debug: Log student_id data for API
        \Log::info('API Registration - ID Number Data:', [
            'raw_student_id' => $data['student_id'] ?? 'null',
            'converted_student_id' => !empty($data['student_id']) ? (int)$data['student_id'] : null,
            'data_type' => gettype($data['student_id'] ?? null)
        ]);

        $user = User::create([
            'role_name' => Role::$user,
            'role_id' => Role::getUserRoleId(),
            $username => $data[$username],
            'student_id' => !empty($data['student_id']) ? (int)$data['student_id'] : null,
            'status' => $disableRegistrationVerificationProcess ? User::$active : User::$pending,
            'password' => Hash::make($data['password']),
            'affiliate' => $usersAffiliateStatus,
            'created_at' => time()
        ]);

        // Debug: Log created user data for API
        \Log::info('API Registration - Created User:', [
            'user_id' => $user->id,
            'student_id' => $user->student_id,
            'student_id_type' => gettype($user->student_id)
        ]);

        if (!empty($data['certificate_additional'])) {
            UserMeta::updateOrCreate([
                'user_id' => $user->id,
                'name' => 'certificate_additional'
            ], [
                'value' => $data['certificate_additional']
            ]);
        }

        $form = $this->getFormFieldsByType($request->get('account_type'));
        $errors = [];

        if (!empty($form)) {
            $fieldErrors = $this->checkFormRequiredFields($request, $form);

            if (!empty($fieldErrors) and count($fieldErrors)) {
                foreach ($fieldErrors as $id => $error) {
                    $errors[$id] = $error;
                }
            }
        }

        if (count($errors)) {
            return apiResponse2(0, 'login', trans('api.auth.login'), $errors);
        }

        $this->storeFormFields($data, $user);

        $verificationController = new VerificationController();
        $verificationController->checkConfirmed($user, $username, $verificationValue);


        return apiResponse2('1', 'stored', trans('api.public.stored'), [
            'user_id' => $user->id
        ]);
    }

    private function stepTwo(Request $request)
    {
        $data = $request->all();
        validateParam($data, [
            'user_id' => 'required|exists:users,id',
            //  'code'=>
        ]);

        $user = User::find($data['user_id']);
        $verificationController = new VerificationController();
        $ee = $user->email ?? $user->mobile;
        return $verificationController->confirmCode($request, $ee);
    }

    private function stepThree(Request $request)
    {
        $data = $request->all();
        $userId = $request->input('user_id');
        
        validateParam($request->all(), [
            'user_id' => 'required|exists:users,id',
            'full_name' => 'required|string|min:3',
            // Made optional per Apple's data collection policies
            'student_id' => 'nullable|numeric|digits_between:8,15|unique:users,student_id,' . $userId,
            'referral_code' => 'nullable|exists:affiliates_codes,code'

        ]);

        $user = User::find($userId);
        // Debug: Log step 3 student_id data
        \Log::info('API Step 3 - ID Number Data:', [
            'raw_student_id' => $data['student_id'] ?? 'null',
            'converted_student_id' => !empty($data['student_id']) ? (int)$data['student_id'] : null,
            'data_type' => gettype($data['student_id'] ?? null)
        ]);

        $user->update([
            'full_name' => $data['full_name'],
            'student_id' => !empty($data['student_id']) ? (int)$data['student_id'] : null
        ]);

        // Debug: Log updated user data
        \Log::info('API Step 3 - Updated User:', [
            'user_id' => $user->id,
            'student_id' => $user->student_id,
            'student_id_type' => gettype($user->student_id)
        ]);

        $enableRegistrationBonus = false;
        $registrationBonusAmount = null;
        $registrationBonusSettings = getRegistrationBonusSettings();
        if (!empty($registrationBonusSettings['status']) and !empty($registrationBonusSettings['registration_bonus_amount'])) {
            $enableRegistrationBonus = true;
            $registrationBonusAmount = $registrationBonusSettings['registration_bonus_amount'];
        }


        $user->update([
            'enable_registration_bonus' => $enableRegistrationBonus,
            'registration_bonus_amount' => $registrationBonusAmount,
        ]);

        $registerReward = RewardAccounting::calculateScore(Reward::REGISTER);
        RewardAccounting::makeRewardAccounting($user->id, $registerReward, Reward::REGISTER, $user->id, true);
        $registrationBonusAccounting = new RegistrationBonusAccounting();
        $registrationBonusAccounting->storeRegistrationBonusInstantly($user);
        $referralCode = $request->input('referral_code', null);
        if (!empty($referralCode)) {
            Affiliate::storeReferral($user, $referralCode);
        }
        event(new Registered($user));
        $token = auth('api')->tokenById($user->id);
        $data['token'] = $token;
        $data['user_id'] = $user->id;
        return apiResponse2(1, 'login', trans('api.auth.login'), $data);

    }

    public function username()
    {
        $email_regex = "/^[_a-z0-9-]+(\.[_a-z0-9-]+)*@[a-z0-9-]+(\.[a-z0-9-]+)*(\.[a-z]{2,})$/i";

        $data = request()->all();

        if (empty($this->username)) {
            if (in_array('mobile', array_keys($data))) {
                $this->username = 'mobile';
            } else if (in_array('email', array_keys($data))) {
                $this->username = 'email';
            }
        }

        return $this->username ?? '';
    }

    public function checkStudentId(Request $request)
    {
        $data = $request->all();
        
        // Check if student_id is provided
        if (empty($data['student_id'])) {
            return apiResponse2(0, 'student_id_required', trans('api.auth.student_id_required'), [
                'available' => false
            ]);
        }

        $studentId = $data['student_id'];
        
        // Check if student_id is numeric
        if (!is_numeric($studentId)) {
            return apiResponse2(0, 'student_id_numeric', trans('api.auth.student_id_numeric'), [
                'available' => false
            ]);
        }

        // Check length - must be exactly 10 digits
        $length = strlen($studentId);
        
        if ($length < 8) {
            return apiResponse2(0, 'student_id_too_short', trans('api.auth.student_id_too_short'), [
                'available' => false
            ]);
        }
        
        if ($length > 15) {
            return apiResponse2(0, 'student_id_too_long', trans('api.auth.student_id_too_long'), [
                'available' => false
            ]);
        }
        
        // Check if student_id already exists in users table
        $exists = User::where('student_id', (int)$studentId)
                     ->whereNotNull('student_id')
                     ->exists();
        
        if ($exists) {
            return apiResponse2(0, 'student_id_exists', trans('api.auth.student_id_exists'), [
                'available' => false
            ]);
        }
        
        return apiResponse2(1, 'student_id_available', trans('api.auth.student_id_available'), [
            'available' => true
        ]);
    }


}
