<?php

namespace App\Console\Commands;

use App\Models\Certificate;
use App\Models\CertificateTemplate;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;

class GenerateCertificateImages extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'certificates:generate-images {--type=all : Generate for specific type (quiz, course, bundle, all)} {--limit=100 : Limit number of certificates to process}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Generate images for existing certificates that don\'t have images';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $type = $this->option('type');
        $limit = (int) $this->option('limit');
        
        $this->info("Starting certificate image generation...");
        $this->info("Type: {$type}, Limit: {$limit}");
        
        // Get certificates without images
        $query = Certificate::whereNull('image_path')
            ->orWhere('image_path', '');
            
        if ($type !== 'all') {
            $query->where('type', $type);
        }
        
        $certificates = $query->limit($limit)->get();
        
        if ($certificates->isEmpty()) {
            $this->info("No certificates found without images.");
            return 0;
        }
        
        $this->info("Found {$certificates->count()} certificates to process.");
        
        $successCount = 0;
        $errorCount = 0;
        
        $progressBar = $this->output->createProgressBar($certificates->count());
        $progressBar->start();
        
        foreach ($certificates as $certificate) {
            try {
                $this->generateCertificateImage($certificate);
                $successCount++;
            } catch (\Exception $e) {
                $errorCount++;
                $this->error("\nError generating image for certificate {$certificate->id}: " . $e->getMessage());
            }
            
            $progressBar->advance();
        }
        
        $progressBar->finish();
        $this->newLine();
        
        $this->info("Generation completed!");
        $this->info("Success: {$successCount}");
        $this->info("Errors: {$errorCount}");
        
        return 0;
    }
    
    /**
     * Generate image for a single certificate
     */
    private function generateCertificateImage($certificate)
    {
        // Generate HTML based on certificate type
        $html = $this->generateCertificateHtml($certificate);
        
        // Generate image using GD library
        $imagePath = $this->generateCertificateImageFromHtml($html, $certificate->id);
        
        if ($imagePath && file_exists($imagePath)) {
            // Move to permanent storage
            $permanentPath = public_path("store/certificates/certificate_{$certificate->id}.png");
            $permanentDir = dirname($permanentPath);
            
            if (!file_exists($permanentDir)) {
                mkdir($permanentDir, 0755, true);
            }
            
            // Copy file to permanent location
            copy($imagePath, $permanentPath);
            
            // Update certificate record with image path
            $certificate->update([
                'image_path' => "store/certificates/certificate_{$certificate->id}.png"
            ]);
            
            // Clean up temporary file
            if (file_exists($imagePath)) {
                unlink($imagePath);
            }
        }
    }
    
    /**
     * Generate HTML for certificate
     */
    private function generateCertificateHtml($certificate)
    {
        $template = CertificateTemplate::where('status', 'publish')
            ->where('type', $certificate->type)
            ->first();

        if (!$template) {
            return $this->generateDefaultCertificateHtml($certificate);
        }

        $user = $certificate->student;
        $body = $template->body;
        $body = $this->replaceCertificatePlaceholders($body, $certificate, $user);

        return $this->wrapCertificateHtml($body);
    }
    
    /**
     * Generate default certificate HTML if no template exists
     */
    private function generateDefaultCertificateHtml($certificate)
    {
        $user = $certificate->student;
        $courseTitle = '';
        
        if ($certificate->webinar) {
            $courseTitle = $certificate->webinar->title;
        } elseif ($certificate->bundle) {
            $courseTitle = $certificate->bundle->title;
        }

        $html = "
        <div style='text-align: center; padding: 50px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; min-height: 400px; display: flex; flex-direction: column; justify-content: center;'>
            <div style='background: white; color: #333; padding: 40px; border-radius: 15px; box-shadow: 0 10px 30px rgba(0,0,0,0.3);'>
                <h1 style='color: #667eea; font-size: 32px; margin-bottom: 30px; font-family: Arial, sans-serif;'>Certificate of Completion</h1>
                <p style='font-size: 18px; margin-bottom: 20px; color: #666;'>This is to certify that</p>
                <h2 style='color: #333; font-size: 28px; margin-bottom: 30px; font-family: Arial, sans-serif; font-weight: bold;'>{$user->full_name}</h2>
                <p style='font-size: 16px; margin-bottom: 20px; color: #666;'>has successfully completed the course</p>
                <h3 style='color: #667eea; font-size: 22px; margin-bottom: 30px; font-family: Arial, sans-serif;'>{$courseTitle}</h3>
                <div style='border-top: 2px solid #667eea; padding-top: 20px; margin-top: 30px;'>
                    <p style='font-size: 14px; color: #666; margin: 5px 0;'>Certificate ID: {$certificate->id}</p>
                    <p style='font-size: 14px; color: #666; margin: 5px 0;'>Issued on: " . date('F j, Y', strtotime($certificate->created_at)) . "</p>
                </div>
            </div>
        </div>";

        return $html;
    }
    
    /**
     * Replace placeholders in certificate template
     */
    private function replaceCertificatePlaceholders($body, $certificate, $user)
    {
        $platformName = getGeneralSettings("site_name");
        $courseTitle = '';

        if ($certificate->webinar) {
            $courseTitle = $certificate->webinar->title;
        } elseif ($certificate->bundle) {
            $courseTitle = $certificate->bundle->title;
        }

        $replacements = [
            '[student]' => $user->full_name,
            '[student_name]' => $user->full_name,
            '[platform_name]' => $platformName,
            '[course]' => $courseTitle,
            '[course_name]' => $courseTitle,
            '[grade]' => $certificate->user_grade ?? '',
            '[certificate_id]' => $certificate->id,
            '[date]' => dateTimeFormat($certificate->created_at, 'j M Y | H:i'),
            '[instructor_name]' => $certificate->webinar ? $certificate->webinar->teacher->full_name : ($certificate->bundle ? $certificate->bundle->teacher->full_name : ''),
            '[duration]' => $certificate->webinar ? $certificate->webinar->duration : ($certificate->bundle ? $certificate->bundle->duration : ''),
        ];

        foreach ($replacements as $placeholder => $value) {
            $body = str_replace($placeholder, $value, $body);
        }

        return $body;
    }
    
    /**
     * Wrap certificate HTML with proper styling
     */
    private function wrapCertificateHtml($body)
    {
        return "
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset='UTF-8'>
            <style>
                body {
                    margin: 0;
                    padding: 0;
                    font-family: Arial, sans-serif;
                    background: white;
                }
                .certificate-container {
                    width: 800px;
                    height: 600px;
                    margin: 0 auto;
                    background: white;
                    position: relative;
                }
            </style>
        </head>
        <body>
            <div class='certificate-container'>
                {$body}
            </div>
        </body>
        </html>";
    }
    
    /**
     * Generate certificate image from HTML using GD library
     */
    private function generateCertificateImageFromHtml($html, $certificateId)
    {
        try {
            // Create image using GD library
            $width = 800;
            $height = 600;
            $image = imagecreatetruecolor($width, $height);
            
            // Set colors
            $white = imagecolorallocate($image, 255, 255, 255);
            $black = imagecolorallocate($image, 0, 0, 0);
            $blue = imagecolorallocate($image, 102, 126, 234);
            $purple = imagecolorallocate($image, 118, 75, 162);
            
            // Fill background with gradient
            $this->createGradientBackground($image, $width, $height, $blue, $purple);
            
            // Extract text content from HTML
            $textContent = strip_tags($html);
            
            // Add certificate content
            $this->addCertificateContent($image, $textContent, $certificateId);
            
            // Save image
            $outputPath = storage_path('app/temp/certificate_' . $certificateId . '.png');
            $tempDir = dirname($outputPath);
            
            if (!file_exists($tempDir)) {
                mkdir($tempDir, 0755, true);
            }
            
            imagepng($image, $outputPath);
            imagedestroy($image);
            
            return file_exists($outputPath) ? $outputPath : false;
            
        } catch (\Exception $e) {
            \Log::error('GD certificate generation failed: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Create gradient background
     */
    private function createGradientBackground($image, $width, $height, $color1, $color2)
    {
        for ($y = 0; $y < $height; $y++) {
            $ratio = $y / $height;
            $r = (int)((1 - $ratio) * 102 + $ratio * 118);
            $g = (int)((1 - $ratio) * 126 + $ratio * 75);
            $b = (int)((1 - $ratio) * 234 + $ratio * 162);
            
            $color = imagecolorallocate($image, $r, $g, $b);
            imageline($image, 0, $y, $width, $y, $color);
        }
    }
    
    /**
     * Add certificate content to image
     */
    private function addCertificateContent($image, $textContent, $certificateId)
    {
        $white = imagecolorallocate($image, 255, 255, 255);
        $black = imagecolorallocate($image, 0, 0, 0);
        $blue = imagecolorallocate($image, 102, 126, 234);
        
        // Create white background rectangle for content
        imagefilledrectangle($image, 50, 50, 750, 550, $white);
        
        // Add title
        $title = "Certificate of Completion";
        $font = public_path('assets/default/fonts/Montserrat-Medium.ttf');
        
        if (file_exists($font)) {
            imagettftext($image, 28, 0, 200, 120, $blue, $font, $title);
            
            // Add certificate ID
            $certId = "Certificate ID: " . $certificateId;
            imagettftext($image, 16, 0, 200, 160, $black, $font, $certId);
            
            // Add date
            $date = "Issued: " . date('F j, Y');
            imagettftext($image, 14, 0, 200, 190, $black, $font, $date);
            
            // Add main content (truncated for GD limitations)
            $lines = explode("\n", wordwrap($textContent, 50));
            $y = 250;
            foreach (array_slice($lines, 0, 8) as $line) {
                imagettftext($image, 12, 0, 100, $y, $black, $font, trim($line));
                $y += 30;
            }
        } else {
            // Fallback to built-in fonts
            imagestring($image, 5, 200, 120, $title, $blue);
            imagestring($image, 3, 200, 160, "Certificate ID: " . $certificateId, $black);
            imagestring($image, 3, 200, 190, "Issued: " . date('F j, Y'), $black);
        }
    }
} 